/**
 *  Copyright (c) 2012 BMW
 *
 *  \author Christian Mueller, christian.ei.mueller@bmw.de BMW 2011,2012
 *
 *  \copyright
 *  Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction,
 *  including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so,
 *  subject to the following conditions:
 *  The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 *  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 *  THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *  For further information see http://www.genivi.org/.
 */

/************************************************************************
 *FILE:      CAmRoutingSenderAsync.cpp
 *SW-COMPONENT:    G3G Audio Component
 *DESCRIPTION:    Implementation of GENIVI IAmRoutingSend Interface.
 *AUTHOR:    RBEI/ECG4 - Bipin Krishnan
 *COPYRIGHT      :    (C) 2013 RBEI
 *HISTORY:
 *06.09.2012    Rev 1.0    RBEI/ECG4 - Bipin Krishnan  Initial Revision
 *10.10.2012    Rev 2.0       RBEI/ECG4 - Bipin Krishnan  Integrated the change for
 *                                                    Serialization of Requests from Audio Manager.
 *23.10.2012      Rev 3.0   RBEI/ECG4 - Bipin Krishnan  Added the Logic for Proper Thread Termination.
 *12.03.2013    Rev 4.0   RBEI/ECG4 - Pratheesh       Adapted to the new architecture
 *                                                    where the control logic is in v850
 *10.04.2013    Rev 5.0   RBEI/ECG4 - Pratheesh       Changed the source list and sink list handling
                                                      Changes done after testing with V850 simulator
 *24.04.2013    Rev 6.0   RBEI/ECG4 - Pratheesh       AM and Vehicle domain connection ID mapping is removed
 *18.09.2013    Rev 7.0   RBEI/ECG4 - Pratheesh       Bypassing AmRoutingPlugin
************************************************************************/
#include <cstdarg>
#include "CAmDltWrapper.h"
#include "CAmRoutingSenderAsync.h"
//#include "projecttypes.h"

using namespace am;

DLT_DECLARE_CONTEXT(RoutingPluginVehicle)

/************************************************************************
 *FUNCTION:  g3g_amroutingplugin_soFactory
 *DESCRIPTION:  This factory method creates an instance of CAmRoutingSenderAsync.
    This is invoked by Audio Manager Daemon.
 *PARAMETER:
 *RETURN VALUE: IAmRoutingSend*
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG4 - Bipin Krishnan  Initial Revision
************************************************************************/
extern "C" IAmRoutingSend* g3g_amroutingplugin_soFactory()
{
    return (new CAmRoutingSenderAsync());
}


/************************************************************************
 *FUNCTION:  destroyRoutingPluginInterfaceAsync
 *DESCRIPTION:  This method deallocates routingSendInterface.
 *PARAMETER:   routingSendInterface: (->I) Pointer to Routing Send Interface.
 *RETURN VALUE: void
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG4 - Bipin Krishnan  Initial Revision
************************************************************************/
extern "C" void destroyRoutingPluginInterfaceAsync(IAmRoutingSend* routingSendInterface)
{
    delete routingSendInterface;
}

/************************************************************************
 *FUNCTION:  CAmRoutingSenderAsync
 *DESCRIPTION:  This constructor initializes Receive Interface, Domains, Sinks,
    Sources, etc. of CAmRoutingSenderAsync object.
    It also creates a Worker Thread to process the Requests from Audio Manager Daemon.
 *PARAMETER:
 *RETURN VALUE: NULL
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG4 - Bipin Krishnan  Initial Revision
 *10.10.2012    Rev 2.0   RBEI/ECG4 - Bipin Krishnan      Creation of One Worker Thread .
 *19.03.2013  Rev 3.0   RBEI/ECG4 - Pratheesh        Changes for RoutingPlugin - Vehicle Domain
************************************************************************/
CAmRoutingSenderAsync::CAmRoutingSenderAsync():g_domainId(0),mReceiveInterface(0),m_pSocketHandler(0),mReceiverShadowInterface(0)
{

   CAmDltWrapper::instance()->registerContext(RoutingPluginVehicle, "RPVD", "RoutingPlugin for vehicle Domain");
   log(&RoutingPluginVehicle,DLT_LOG_INFO,"CAmRoutingSenderAsync Constructor Called");

}

/************************************************************************
 *FUNCTION:  ~CAmRoutingSenderAsync
 *DESCRIPTION:  Destructor of CAmRoutingSenderAsync.
 *PARAMETER:
 *RETURN VALUE: void
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG4 - Bipin Krishnan  Initial Revision
 *10.10.2012    Rev 2.0   RBEI/ECG4 - Bipin Krishnan      Added the Thread Join for Worker Thread.
 *23.10.2012  Rev 3.0   RBEI/ECG4 - Bipin Krishnan  Removed the Thread Join from this method for Proper Thread Termination.
 *19.03.2013  Rev 4.0   RBEI/ECG4 - Pratheesh        Changes for RoutingPlugin - Vehicle Domain
************************************************************************/
CAmRoutingSenderAsync::~CAmRoutingSenderAsync() throw()
{
  if(mReceiverShadowInterface)
  {
    delete mReceiverShadowInterface;
    mReceiverShadowInterface = NULL;
  }
  mReceiveInterface = NULL;
  m_pSocketHandler = NULL;
}

/************************************************************************
 *FUNCTION:  startupInterface
 *DESCRIPTION:  This is the initialization routine invoked by Audio Manager Daemon after loading routing
    plugin.
 *PARAMETER:   routingreceiveinterface: (->I) Routing Receive Interface passed by Audio Manager Daemon.
 *RETURN VALUE: am_Error_e
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG4 - Bipin Krishnan  Initial Revision
 *10.10.2012    Rev 2.0   RBEI/ECG4 - Bipin Krishnan      Creation of startUpJob and adding in Job Queue.
 *19.03.2013  Rev 3.0   RBEI/ECG4 - Pratheesh        Changes for RoutingPlugin - Vehicle Domain
************************************************************************/
am_Error_e CAmRoutingSenderAsync::startupInterface(IAmRoutingReceive *routingreceiveinterface)
{
  log(&RoutingPluginVehicle,DLT_LOG_INFO,"startupInterface Called");
  assert(routingreceiveinterface != NULL);
  am_Error_e enErr = E_OK;

      mReceiveInterface = routingreceiveinterface;

  if(NULL != mReceiveInterface)
    mReceiveInterface->getSocketHandler(m_pSocketHandler);

  //create the RoutingReceiverShadow
    mReceiverShadowInterface = new IAmRoutingReceiverShadow(mReceiveInterface, m_pSocketHandler);

  return enErr;
}
/************************************************************************
 *FUNCTION     : enRegisterDomain
 *DESCRIPTION  : Register the Vehicle domain data with Audio Manager
 *PARAMETER    : none

 *RETURN VALUE: am_Error_e
 *HISTORY         :
 *10.04.2013   Rev 1.0     RBEI/ECG4 - Pratheesh   Initial Revision
************************************************************************/
am_Error_e CAmRoutingSenderAsync::enRegisterDomain()
{
   log(&RoutingPluginVehicle,DLT_LOG_INFO,"enRegisterDomain ");
   am_Error_e enErr = E_UNKNOWN;
   am_Domain_s rDomainData;

   rDomainData.busname = BUS_NAME;
   rDomainData.domainID = 0;
   rDomainData.name = DOMAIN_NAME;
   rDomainData.early = false;
   rDomainData.nodename = NODE_NAME;
   rDomainData.state = DS_CONTROLLED;

   if(NULL != mReceiverShadowInterface)
      enErr = mReceiverShadowInterface->registerDomain(rDomainData,g_domainId);
   else
      log(&RoutingPluginVehicle,DLT_LOG_INFO,"enRegisterDomain:mReceiverShadowInterface is NULL");

   log(&RoutingPluginVehicle,DLT_LOG_INFO,"enRegisterDomain:Vehicle Domain Registered with ID:",g_domainId,",Error Code:",enErr);

   return enErr;
}

/************************************************************************
 *FUNCTION:  setRoutingReady
 *DESCRIPTION:  This method is invoked by Audio Manager Daemon when Controller Plugin
    intimates setRoutingReady to this Daemon.
 *PARAMETER:   handle: (I) handle of set routing ready request.
 *RETURN VALUE: void
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG4 - Bipin Krishnan  Initial Revision
 *10.10.2012    Rev 2.0    RBEI/ECG4 - Bipin Krishnan      Creation of syncRegisterJob and
                adding in the Job Queue.
 *19.03.2013  Rev 3.0   RBEI/ECG4 - Pratheesh        Changes for RoutingPlugin - Vehicle Domain
 *19.06.2013  Rev 4.0   RBEI/ECG4 - Vyankatesh      Changes for RoutingPlugin - Nisan Porting
************************************************************************/
void CAmRoutingSenderAsync::setRoutingReady(const uint16_t handle)
{
   assert(handle!=0);

   log(&RoutingPluginVehicle,DLT_LOG_INFO,"setRoutingReady Called with handle",handle);

   if (E_OK == enRegisterDomain())
   {
     log(&RoutingPluginVehicle,DLT_LOG_INFO,"setRoutingReady: enRegisterDomain returned successfully");
   }
   else
   {
     log(&RoutingPluginVehicle,DLT_LOG_INFO,"setRoutingReady: enRegisterDomain failed");
   }
   log(&RoutingPluginVehicle,DLT_LOG_INFO,"setRoutingReady: calling mReceiverShadowInterface->confirmRoutingReady");

   if(mReceiverShadowInterface)
   {
  log(&RoutingPluginVehicle,DLT_LOG_INFO,"CAmRoutingSenderAsync::setRoutingReady ,Calling confirmRoutingReady");
  mReceiverShadowInterface->confirmRoutingReady(handle,E_OK);
   }
}

/************************************************************************
 *FUNCTION:  setRoutingRundown
 *DESCRIPTION:  This method is invoked by Audio Manager Daemon during Rundown
    of the Audio Management service.
 *PARAMETER:   handle: (I) handle of set routing rundown request.
 *RETURN VALUE: void
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG4 - Bipin Krishnan  Initial Revision
 *10.10.2012  Rev 2.0    RBEI/ECG4 - Bipin Krishnan      Removed the Logic of checking the Acknowledgement
                  from ADR for Mute Command issued.
 *23.10.2012    Rev 3.0   RBEI/ECG4 - Bipin Krishnan  Added the Logic for Proper Thread Termination.
 *19.03.2013  Rev 4.0   RBEI/ECG4 - Pratheesh        Changes for RoutingPlugin - Vehicle Domain
************************************************************************/
void CAmRoutingSenderAsync::setRoutingRundown(const uint16_t handle)
{
  assert(handle!=0);

  log(&RoutingPluginVehicle,DLT_LOG_INFO,"setRoutingRundown");

  if(NULL != mReceiverShadowInterface)
    mReceiverShadowInterface->confirmRoutingRundown(handle, E_OK);//modified
}

/************************************************************************
 *FUNCTION    :  asyncAbort (NOT USED AS OF NOW)
 *DESCRIPTION  :  This method is invoked by Audio Manager Daemon to cancel the ongoing operations like volume change, connect request.
 *PARAMETER    : handle: (I) handle of the operations(eg: volume change, connect request).
 *RETURN VALUE: am_Error_e
 *HISTORY      :
 *06.09.2012  Rev 1.0    RBEI/ECG4 - Bipin Krishnan  Initial Revision
 *10.10.2012  Rev 2.0    RBEI/ECG4 - Bipin Krishnan  Commented the cancelWork on ThreadPool.
 *19.03.2013  Rev 3.0   RBEI/ECG4 - Pratheesh        Changes for RoutingPlugin - Vehicle Domain
************************************************************************/
am_Error_e CAmRoutingSenderAsync::asyncAbort(const am_Handle_s /*handle*/)
{
    return (E_NOT_USED);
}

/************************************************************************
 *FUNCTION:  asyncConnect
 *DESCRIPTION:  This method is invoked by Audio Manager Daemon to connect a source and a sink.

 *PARAMETER:   handle: (I) handle of the connection request.
    connectionID: (I) connection identifier.
    sourceID: (I) source identifier.
    sinkID: (I) sink identifier.
    connectionFormat: (I) connection format.

 *RETURN VALUE: am_Error_e
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG4 - Bipin Krishnan  Initial Revision
 *10.10.2012  Rev 2.0   RBEI/ECG4 - Bipin Krishnan   Creation of asyncConnectJob and
                  Adding the same in Job Queue.
 *19.03.2013  Rev 3.0   RBEI/ECG4 - Pratheesh        Changes for RoutingPlugin - Vehicle Domain
 *23.04.2015  Rev 4.0   RBEI/ECV3 - Ranjit           Removed asserts : SUZUKI-23543
************************************************************************/
am_Error_e CAmRoutingSenderAsync::asyncConnect(const am_Handle_s handle, const am_connectionID_t connectionID, const am_sourceID_t sourceID, const am_sinkID_t sinkID, const am_CustomConnectionFormat_t /*connectionFormat*/)
{
   uint16_t u16Handle = handle.handle;
   log(&RoutingPluginVehicle,DLT_LOG_INFO,"asyncConnect Called, Handle", u16Handle,
             "Connection Id",connectionID, "Source Id",sourceID,
             "Sink Id",sinkID);

   if(mReceiverShadowInterface){
     log(&RoutingPluginVehicle,DLT_LOG_INFO,"asyncConnect, Calling ackConnect");
     mReceiverShadowInterface->ackConnect(handle,connectionID,E_OK);
   }

   return (E_OK);
}

/************************************************************************
 *FUNCTION:  asyncDisconnect
 *DESCRIPTION:  This method is invoked by Audio Manager Daemon to disconnect an existing connection between a source and a sink.
 *PARAMETER:   handle: (I) handle
    connectionID: (I) connection identifier.
 *RETURN VALUE: am_Error_e
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG4 - Bipin Krishnan  Initial Revision
 *10.10.2012  Rev 2.0    RBEI/ECG4 - Bipin Krishnan      Creation of asyncDisConnectJob and
                  adding the same in Job Queue.
 *19.03.2013  Rev 3.0   RBEI/ECG4 - Pratheesh        Changes for RoutingPlugin - Vehicle Domain
************************************************************************/
am_Error_e CAmRoutingSenderAsync::asyncDisconnect(const am_Handle_s handle, const am_connectionID_t connectionID)
{

  log(&RoutingPluginVehicle,DLT_LOG_INFO,"asyncDisconnect Called, Handle", handle.handle,
    "Connection Id",connectionID);

  if(mReceiverShadowInterface){
    mReceiverShadowInterface->ackDisconnect(handle,connectionID,E_OK);
  }

  return (E_OK); //Shortcut
}

/************************************************************************
 *FUNCTION:  asyncSetSinkVolume
 *DESCRIPTION:  This method is invoked by Audio Manager Daemon to set the volume of a sink.
    It even handles Mute/De-Mute of a sink.
 *PARAMETER:   handle: (I)  handle.
    sinkID:(I) Sink Identifier.
    volume: (I) Volume
    ramp: (I) Ramp
    time: (I) Threshold time for Setting the volume.

 *RETURN VALUE: am_Error_e
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG4 - Bipin Krishnan  Initial Revision
 *10.10.2012  Rev 2.0    RBEI/ECG4 - Bipin Krishnan      Creation of asyncSetSinkVolumeJob and
                  Adding the same in the Job Queue.
 *19.03.2013  Rev 3.0   RBEI/ECG4 - Pratheesh        Changes for RoutingPlugin - Vehicle Domain
************************************************************************/
am_Error_e CAmRoutingSenderAsync::asyncSetSinkVolume(const am_Handle_s handle, const am_sinkID_t sinkID, const am_volume_t volume, const am_CustomRampType_t , const am_time_t amTimeData)
{

  log(&RoutingPluginVehicle,DLT_LOG_INFO,"asyncSetSinkVolume Called, Handle", handle.handle,
            "SinkID",sinkID,"Volume",volume,"time", amTimeData);

  return (E_OK);
}

/************************************************************************
 *FUNCTION    : asyncSetSourceSoundProperty
 *DESCRIPTION  : To set the sound property of a source
 *PARAMETER    : handle (I) handle
                sourceID (I) source identifier
                soundProperty (I) structure with property type and value to be set

 *RETURN VALUE: am_Error_e
 *HISTORY      :
 *10.04.2013  Rev 1.0    RBEI/ECG4 - Pratheesh  Initial Revision
************************************************************************/
am_Error_e CAmRoutingSenderAsync::asyncSetSourceSoundProperty(const am_Handle_s handle, const am_sourceID_t sourceID, const am_SoundProperty_s & soundProperty)
{

  log(&RoutingPluginVehicle,DLT_LOG_INFO,"asyncSetSourceSoundProperty Called, Handle", handle.handle,
                "SourceID",sourceID,"soundProperty",soundProperty.type,  "Value", soundProperty.value);

  switch(soundProperty.type)
  {
    case SP_SOUND_TREBLE :
    {
      log(&RoutingPluginVehicle,DLT_LOG_INFO,"asyncSetSourceSoundProperty Called for SP_SOUND_TREBLE");
      break;
    }
    case SP_SOUND_MIDDLE :
    {
      log(&RoutingPluginVehicle,DLT_LOG_INFO,"asyncSetSourceSoundProperty Called for SP_SOUND_MID");
      break;
    }
    case SP_SOUND_BASS :
    {
      log(&RoutingPluginVehicle,DLT_LOG_INFO,"asyncSetSourceSoundProperty Called for SP_SOUND_BASS");
      break;
    }
    default :
    {
      log(&RoutingPluginVehicle,DLT_LOG_INFO,"asyncSetSourceSoundProperty Called for unknown...");
      break;
    }
  }

  return (E_OK);
}

/************************************************************************
 *FUNCTION:  returnBusName
 *DESCRIPTION:  This method is invoked by Audio Manager Daemon to return the bus name used in
    routing plugin. Later the plugin will be referred in Audio Manager by this bus name.
 *PARAMETER:   busName: (IO) Bus Name.
 *RETURN VALUE: am_Error_e
 *HISTORY:
 *06.09.2012  Rev 1.0    RBEI/ECG4 - Bipin Krishnan  Initial Revision
 *19.03.2013  Rev 2.0   RBEI/ECG4 - Pratheesh        Changes for RoutingPlugin - Vehicle Domain
************************************************************************/
am_Error_e CAmRoutingSenderAsync::returnBusName(std::string & BusName) const
{
  log(&RoutingPluginVehicle,DLT_LOG_INFO,"returnBusName Called");
    BusName = BUS_NAME;
    return (E_OK);
}

/************************************************************************
 *FUNCTION:  getInterfaceVersion
 *DESCRIPTION:  This method is used to fetch the version of routing send interface.
 *PARAMETER:   version: (IO) version of routing send interface.
 *RETURN VALUE: void
 *HISTORY      :
 *06.09.2012  Rev 1.0    RBEI/ECG4 - Bipin Krishnan  Initial Revision
 *19.03.2013  Rev 2.0   RBEI/ECG4 - Pratheesh        Changes for RoutingPlugin - Vehicle Domain
************************************************************************/
void CAmRoutingSenderAsync::getInterfaceVersion(std::string & version) const
{
  log(&RoutingPluginVehicle,DLT_LOG_INFO,"getInterfaceVersion Called");

    //RoutingSendVersion is defined to 1.0 in the header IAmRoutingSend.h
    version = RoutingVersion;
}

am_Error_e CAmRoutingSenderAsync::asyncSetSourceVolume(const am_Handle_s, const am_sourceID_t, const am_volume_t, const am_CustomRampType_t , const am_time_t )
{
  return (E_NOT_USED);
}

am_Error_e CAmRoutingSenderAsync::asyncSetSourceState(const am_Handle_s handle, const am_sourceID_t , const am_SourceState_e)
{
   log(&RoutingPluginVehicle,DLT_LOG_INFO,"asyncSetSourceState: calling ackSetSourceState");
   if(mReceiveInterface)
     mReceiveInterface->ackSetSourceState(handle, E_OK);
  return(E_OK);
}

am_Error_e CAmRoutingSenderAsync::asyncSetSinkSoundProperty(const am_Handle_s handle, const am_sinkID_t sinkID, const am_SoundProperty_s & soundProperty)
{
  (void) sinkID;
  (void) soundProperty;
  if(mReceiverShadowInterface){
    log(&RoutingPluginVehicle,DLT_LOG_INFO,"asyncSetSinkSoundProperty Called,returning E_OK");
    mReceiverShadowInterface->ackSetSinkSoundProperty(handle,E_OK);
  }
  return(E_OK);
}

am_Error_e CAmRoutingSenderAsync::asyncCrossFade(const am_Handle_s , const am_crossfaderID_t , const am_HotSink_e , const am_CustomRampType_t , const am_time_t )
{
  return (E_NOT_USED);
}
am_Error_e CAmRoutingSenderAsync::setDomainState(const am_domainID_t , const am_DomainState_e )
{
  return (E_NOT_USED);
}

am_Error_e CAmRoutingSenderAsync::asyncSetSourceSoundProperties(const am_Handle_s , const am_sourceID_t , const std::vector<am_SoundProperty_s> &)
{
    return (E_NOT_USED);
}

am_Error_e CAmRoutingSenderAsync::asyncSetSinkSoundProperties(const am_Handle_s , const am_sinkID_t , const std::vector<am_SoundProperty_s> & )
{
    return (E_NOT_USED);
}
am_Error_e am::CAmRoutingSenderAsync::asyncSetVolumes(const am_Handle_s handle, const std::vector<am_Volumes_s>& listVolumes)
{
    (void)handle;
    (void)listVolumes;
    //todo: implement
    return (E_NOT_USED);
}

am_Error_e am::CAmRoutingSenderAsync::asyncSetSinkNotificationConfiguration(const am_Handle_s handle, const am_sinkID_t sinkID, const am_NotificationConfiguration_s& notificationConfiguration)
{
    (void)handle;
    (void)sinkID;
    (void)notificationConfiguration;
    //todo: implement
    return (E_NOT_USED);
}

am_Error_e am::CAmRoutingSenderAsync::asyncSetSourceNotificationConfiguration(const am_Handle_s handle, const am_sourceID_t sourceID, const am_NotificationConfiguration_s& notificationConfiguration)
{
    (void)handle;
    (void)sourceID;
    (void)notificationConfiguration;
    //todo: implement
    return (E_NOT_USED);
}
